/*
 * ALSA driver for Panasonic UniPhier series.
 * 
 * Copyright (c) 2013 Panasonic corporation.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#include <linux/irqshdrv.h>

#include "mn2ws-pcm.h"

MODULE_AUTHOR("Katsuhiro Suzuki <suzuki.katsuhiro002@jp.panasonic.com>");
MODULE_DESCRIPTION("Panasonic UniPhier MN2WS0230 PCM Driver(AIO direct)");
MODULE_LICENSE("GPL");
MODULE_SUPPORTED_DEVICE(
	"{{Panasonic, MN2WS0230}, "
	"{Panasonic, MN2WS0230x}}");


#define MN2WS_PCM_AIO_DEV_NAME       "mn2ws0230-pcm-aio"
#define MN2WS_PCM_AIO_DEV_COUNT      3

#define MN2WS_RES_PCM            0
#define MN2WS_RES_AIO2           1

#define MN2WS_PCM_REG_START      (0x5e000000)
#define MN2WS_PCM_REG_SIZE       (0x00013000)
#define MN2WS_AIO2_REG_START     (0x56000000)
#define MN2WS_AIO2_REG_SIZE      (0x00050000)

//tentative: use 0x4000 + 0x4000 only(restricts of HW)
#define MN2WS_PCM_HW_BUF_SIZE         (PAGE_SIZE * 8)
//tentative: use 0x10000 + 0x4000 only(restricts of HW)
#define MN2WS_PCM_HW_BUF_SIZE_HDMI    (PAGE_SIZE * 20)

//tentative: use 0x4000 only(restricts of HW)
#define MN2WS_PCM_HW_USE_SIZE         (PAGE_SIZE * 4)
//tentative: use 0x10000 only(restricts of HW)
#define MN2WS_PCM_HW_USE_SIZE_HDMI    (PAGE_SIZE * 16)


//Dummy channel to generate internal clock
#define MN2WS_PCM_AIO_DUMMY_CH_MAIN      (3)
#define MN2WS_PCM_AIO_DUMMY_CH_IEC       (4)
#define MN2WS_PCM_AIO_DUMMY_CH_HDMI      (5)

//tentative: use 0x800 only(restricts of HW)
#define MN2WS_PCM_AIO_CHUNK_SIZE         (0x800)
//tentative: use 0x2000 only(restricts of HW)
#define MN2WS_PCM_AIO_CHUNK_SIZE_HDMI    (0x2000)


//APCMOUT
#define MN2WS_APCMOUT_MODE(n)             (0x5e011730 + 0x30 * (n))
#define MN2WS_APCMOUT_SEMAPH(n)           (0x5e011734 + 0x30 * (n))
#define MN2WS_APCMOUT_DEVSEL_CONF(n)      (0x5e011738 + 0x30 * (n))
#define MN2WS_APCMOUT_DEVSEL_SEND(n)      (0x5e01173c + 0x30 * (n))
#define MN2WS_APCMOUT_SAMP(n)             (0x5e011740 + 0x30 * (n))

#define MN2WS_APCMOUT_SEND_STADDR(n)      (0x5e011b30 + 0x30 * (n))
#define MN2WS_APCMOUT_SEND_ENDADDR(n)     (0x5e011b34 + 0x30 * (n))
#define MN2WS_APCMOUT_SEND_RDSRC(n)       (0x5e011b38 + 0x30 * (n))
#define MN2WS_APCMOUT_SEND_WRSRC(n)       (0x5e011b3c + 0x30 * (n))
#define MN2WS_APCMOUT_SEND_XOCTL(n)       (0x5e011b40 + 0x30 * (n))
#define MN2WS_APCMOUT_SEND_XOFS(n)        (0x5e011b44 + 0x30 * (n))

#define MN2WS_APCMOUT_SEND_HDMIPCMOCTL    (0x5e011ba8)
#define MN2WS_APCMOUT_SEND_HDMIIECOCTL    (0x5e011bac)
#define MN2WS_APCMOUT_SEND_HDMIFMTSEL     (0x5e011bb0)

#define MN2WS_APCMOUT_SEND_DUMMY0FS       (0x5e011bd0)

//Other
#define MN2WS_INTSTATAVIO6                (0x5e008658)
#define MN2WS_SYSCTRCMDA                  (0x5e00c09c)
#define MN2WS_ADECOUTCTL                  (0x5e010c00)
#define MN2WS_PCMOCTL_0                   (0x5e010f10)
#define MN2WS_IECOCTL_0                   (0x5e010f18)
#define MN2WS_HDMIOCTL_0                  (0x5e010f1c)
#define MN2WS_SRCCTL_0                    (0x5e010f20)
#define MN2WS_SRCCTL_1                    (0x5e010f24)



//SW view CHMap   :0x56000000
#define MN2WS_A2CHNMapCtr0(n)             (0x56000000 + 0x40 * (n))
//SW view RBMap   :0x56001000
#define MN2WS_A2RBNMapCtr0(n)             (0x56001000 + 0x40 * (n))
//SW view IPortMap:0x56002000
#define MN2WS_A2IPortNMapCtr0(n)          (0x56002000 + 0x40 * (n))
#define MN2WS_A2IPortNMapCtr1(n)          (0x56002004 + 0x40 * (n))
//SW view IIFMap  :0x56003000
#define MN2WS_A2IIFNMapCtr0(n)            (0x56003000 + 0x40 * (n))
//SW view OPortMap:0x56004000
#define MN2WS_A2OPortNMapCtr0(n)          (0x56004000 + 0x40 * (n))
#define MN2WS_A2OPortNMapCtr1(n)          (0x56004004 + 0x40 * (n))
#define MN2WS_A2OPortNMapCtr2(n)          (0x56004008 + 0x40 * (n))
//SW view OIFMap  :0x56005000
#define MN2WS_A2OIFNMapCtr0(n)            (0x56005000 + 0x40 * (n))


//AIN(PCMINN) AINPORT_SW:0x56022000
//#define MN2WS_A2SWIPortMX(n)    (0x56022000 + 0x400 * (n))
#define MN2WS_A2SWIPortMXCtr1(n)          (0x56022000 + 0x400 * (n))
#define MN2WS_A2SWIPortMXCtr2(n)          (0x56022004 + 0x400 * (n))
#define MN2WS_A2SWIPortMXACCtr(n)         (0x5602200c + 0x400 * (n))
#define MN2WS_A2SWIPortMXCntCtr(n)        (0x56022010 + 0x400 * (n))
#define MN2WS_A2SWIPortMXCounter(n)       (0x56022014 + 0x400 * (n))
#define MN2WS_A2SWIPortMXACLKSel0Ex(n)    (0x56022020 + 0x400 * (n))
#define MN2WS_A2SWIPortMXIR(n)            (0x5602202c + 0x400 * (n))
#define MN2WS_A2SWIPortMXIM0(n)           (0x56022030 + 0x400 * (n))
#define MN2WS_A2SWIPortMXID0(n)           (0x56022040 + 0x400 * (n))
#define MN2WS_A2SWIPortMXEXNOE(n)         (0x56022070 + 0x400 * (n))
#define MN2WS_A2SWIPortMXRstCtr(n)        (0x5602207c + 0x400 * (n))

//AIN(PBinMX) PBIN_SW:0x56020200
#define MN2WS_A2SWPBintMXCtr(n)           (0x56020200 + 0x40 * (n))


//AOUT(PCMOUTN) AOUTPORT_SW:0x56042000
#define MN2WS_A2SWOPortMXCtr1(n)               (0x56042000 + 0x400 * (n))
#define MN2WS_A2SWOPortMXCtr2(n)               (0x56042004 + 0x400 * (n))
#define MN2WS_A2SWOPortMXCtr3(n)               (0x56042008 + 0x400 * (n))
#define MN2WS_A2SWOPortMXACLKSel0Ex(n)         (0x56042030 + 0x400 * (n))
#define MN2WS_A2SWOPortMXPath(n)               (0x56042040 + 0x400 * (n))
#define MN2WS_A2SWOPortMXSYNC(n)               (0x56042044 + 0x400 * (n))
#define MN2WS_A2SWOPortMXCntCtr(n)             (0x56042060 + 0x400 * (n))
#define MN2WS_A2SWOPortMXCounter(n)            (0x56042064 + 0x400 * (n))
#define MN2WS_A2SWOPortMXIR(n)                 (0x56042070 + 0x400 * (n))
#define MN2WS_A2SWOPortMXIM0(n)                (0x56042080 + 0x400 * (n))
#define MN2WS_A2SWOPortMXIM1(n)                (0x56042084 + 0x400 * (n))
#define MN2WS_A2SWOPortMXIM2(n)                (0x56042088 + 0x400 * (n))
#define MN2WS_A2SWOPortMXIM3(n)                (0x5604208c + 0x400 * (n))
#define MN2WS_A2SWOPortMXID0(n)                (0x56042090 + 0x400 * (n))
#define MN2WS_A2SWOPortMXID1(n)                (0x56042094 + 0x400 * (n))
#define MN2WS_A2SWOPortMXID2(n)                (0x56042098 + 0x400 * (n))
#define MN2WS_A2SWOPortMXID3(n)                (0x5604209c + 0x400 * (n))
#define MN2WS_A2SWOPortMXEXNOE(n)              (0x560420f0 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT0VolPara1(n)         (0x56042100 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT1VolPara1(n)         (0x56042120 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT2VolPara1(n)         (0x56042140 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT3VolPara1(n)         (0x56042160 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT4VolPara1(n)         (0x56042180 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT0VolPara2(n)         (0x56042104 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT1VolPara2(n)         (0x56042124 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT2VolPara2(n)         (0x56042144 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT3VolPara2(n)         (0x56042164 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT4VolPara2(n)         (0x56042184 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT0VolGainStatus(n)    (0x56042108 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT1VolGainStatus(n)    (0x56042128 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT2VolGainStatus(n)    (0x56042148 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT3VolGainStatus(n)    (0x56042168 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT4VolGainStatus(n)    (0x56042188 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT0VolChPara(n)        (0x5604210c + 0x400 * (n))
#define MN2WS_A2SWOPortMXT1VolChPara(n)        (0x5604212c + 0x400 * (n))
#define MN2WS_A2SWOPortMXT2VolChPara(n)        (0x5604214c + 0x400 * (n))
#define MN2WS_A2SWOPortMXT3VolChPara(n)        (0x5604216c + 0x400 * (n))
#define MN2WS_A2SWOPortMXT4VolChPara(n)        (0x5604218c + 0x400 * (n))
#define MN2WS_A2SWOPortMXT0VolCtr(n)           (0x56042110 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT1VolCtr(n)           (0x56042130 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT2VolCtr(n)           (0x56042150 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT3VolCtr(n)           (0x56042170 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT4VolCtr(n)           (0x56042190 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT0SlotCtr(n)          (0x56042114 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT1SlotCtr(n)          (0x56042134 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT2SlotCtr(n)          (0x56042154 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT3SlotCtr(n)          (0x56042174 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT4SlotCtr(n)          (0x56042194 + 0x400 * (n))
#define MN2WS_A2SWOPortMXT0RstCtr(n)           (0x5604211c + 0x400 * (n))
#define MN2WS_A2SWOPortMXT1RstCtr(n)           (0x5604213c + 0x400 * (n))
#define MN2WS_A2SWOPortMXT2RstCtr(n)           (0x5604215c + 0x400 * (n))
#define MN2WS_A2SWOPortMXT3RstCtr(n)           (0x5604217c + 0x400 * (n))
#define MN2WS_A2SWOPortMXT4RstCtr(n)           (0x5604219c + 0x400 * (n))

//AOUT(PBoutMX) PBOUT_SW:0x56040200
#define MN2WS_A2SWPBoutMXCtr0(n)               (0x56040200 + 0x40 * (n))
#define MN2WS_A2SWPBoutMXCtr1(n)               (0x56040204 + 0x40 * (n))
#define MN2WS_A2SWPBoutMXIntCtr(n)             (0x56040208 + 0x40 * (n))


//A2D(subsystem)
#define MN2WS_CDA2D_STRT0                 (0x56010000)
#define MN2WS_CDA2D_PERFCNFG              (0x56010008)
#define MN2WS_CDA2D_STAT0                 (0x56010020)
#define MN2WS_CDA2D_CHMXCTRL1(n)          (0x56011000 + 0x40 * (n))
#define MN2WS_CDA2D_CHMXCTRL2(n)          (0x56011004 + 0x40 * (n))
#define MN2WS_CDA2D_CHMXSTAT(n)           (0x56011010 + 0x40 * (n))
#define MN2WS_CDA2D_CHMXIR(n)             (0x56011014 + 0x40 * (n))
#define MN2WS_CDA2D_CHMXIE(n)             (0x56011018 + 0x40 * (n))
#define MN2WS_CDA2D_CHMXID(n)             (0x5601101c + 0x40 * (n))
#define MN2WS_CDA2D_CHMXSRCAMODE(n)       (0x56011020 + 0x40 * (n))
#define MN2WS_CDA2D_CHMXDSTAMODE(n)       (0x56011024 + 0x40 * (n))
#define MN2WS_CDA2D_CHMXSRCSTRTADRS(n)    (0x56011028 + 0x40 * (n))
#define MN2WS_CDA2D_CHMXDSTSTRTADRS(n)    (0x5601102c + 0x40 * (n))
#define MN2WS_CDA2D_CHMXSIZE(n)           (0x56011030 + 0x40 * (n))

//A2D(ring buffer)
#define MN2WS_CDA2D_RBFLUSH0              (0x56010040)
#define MN2WS_CDA2D_PARTRESET0            (0x56010050)
#define MN2WS_CDA2D_PARTRESET1            (0x56010054)
#define MN2WS_CDA2D_RBMXBGNADRS(n)        (0x56012000 + 0x40 * (n))
#define MN2WS_CDA2D_RBMXENDADRS(n)        (0x56012004 + 0x40 * (n))
#define MN2WS_CDA2D_RBMXBTH(n)            (0x56012008 + 0x40 * (n))
#define MN2WS_CDA2D_RBMXRTH(n)            (0x5601200c + 0x40 * (n))
#define MN2WS_CDA2D_RBMXSTAT(n)           (0x56012010 + 0x40 * (n))
#define MN2WS_CDA2D_RBMXIR(n)             (0x56012014 + 0x40 * (n))
#define MN2WS_CDA2D_RBMXIE(n)             (0x56012018 + 0x40 * (n))
#define MN2WS_CDA2D_RBMXID(n)             (0x5601201c + 0x40 * (n))
#define MN2WS_CDA2D_RBMXRDPTR(n)          (0x56012020 + 0x40 * (n))
#define MN2WS_CDA2D_RBMXWRPTR(n)          (0x56012028 + 0x40 * (n))
#define MN2WS_CDA2D_RBMXCNFG(n)           (0x56012030 + 0x40 * (n))
#define MN2WS_CDA2D_RBMXPTRCTRL(n)        (0x56012034 + 0x40 * (n))


struct mn2ws0230_pcm_aio_pcmif_desc {
	//Dummy channel to generate internal clock
	int dummy_ch;
	//Bit position of ADECOUTCTL, SYSCTRCMDA register
	int decoutctl_bit;
	//Bit position of DEVSEL_CONF, DEVSEL_SEND register
	int devsel_bit;
	size_t chunk_size;
};

struct mn2ws0230_pcm_aio_desc {
	struct mn2ws0230_pcm_aio_pcmif_desc play;
	struct mn2ws0230_pcm_aio_pcmif_desc cap;
};


static int mn2ws0230_pcm_aio_hwdep_force_aout(struct mn2ws_pcm_dev *dev);

static int mn2ws0230_pcm_aio_play_init(struct mn2ws_pcm_dev *dev);
static int mn2ws0230_pcm_aio_play_term(struct mn2ws_pcm_dev *dev);
static int mn2ws0230_pcm_aio_play_hardware(struct mn2ws_pcm_dev *dev, struct snd_pcm_hardware *h);
static int mn2ws0230_pcm_aio_play_hardware_hdmi(struct mn2ws_pcm_dev *dev, struct snd_pcm_hardware *h);
static int mn2ws0230_pcm_aio_play_setup_main(struct mn2ws_pcm_dev *dev);
static int mn2ws0230_pcm_aio_play_setup_iec(struct mn2ws_pcm_dev *dev);
static int mn2ws0230_pcm_aio_play_setup_hdmi(struct mn2ws_pcm_dev *dev);
static int mn2ws0230_pcm_aio_play_setup_mode(struct mn2ws_pcm_dev *dev);
static int mn2ws0230_pcm_aio_play_setup_buffer(struct mn2ws_pcm_dev *dev);
static int mn2ws0230_pcm_aio_play_setup_dma(struct mn2ws_pcm_dev *dev);
static int mn2ws0230_pcm_aio_play_mute(struct mn2ws_pcm_dev *dev);
static int mn2ws0230_pcm_aio_play_start(struct mn2ws_pcm_dev *dev);
static int mn2ws0230_pcm_aio_play_stop(struct mn2ws_pcm_dev *dev);
static unsigned long mn2ws0230_pcm_aio_play_get_hwptr(struct mn2ws_pcm_dev *dev);
static void mn2ws0230_pcm_aio_play_set_devptr(struct mn2ws_pcm_dev *dev, unsigned long ptr);
static int mn2ws0230_pcm_aio_play_waitevent(struct mn2ws_pcm_dev *dev);
static ssize_t mn2ws0230_pcm_aio_play_copy_to_hw(struct mn2ws_pcm_dev *dev, struct ringbuf *r, size_t s);
static ssize_t mn2ws0230_pcm_aio_play_silence_to_hw(struct mn2ws_pcm_dev *dev, size_t s);
static int mn2ws0230_pcm_aio_init_module(void);
static void mn2ws0230_pcm_aio_exit_module(void);


static struct resource mn2ws_resources[] = {
	[MN2WS_RES_PCM] = {
		.start = MN2WS_PCM_REG_START, 
		.end   = MN2WS_PCM_REG_START + MN2WS_PCM_REG_SIZE, 
		.flags = IORESOURCE_MEM, 
	}, 
	[MN2WS_RES_AIO2] = {
		.start = MN2WS_AIO2_REG_START, 
		.end   = MN2WS_AIO2_REG_START + MN2WS_AIO2_REG_SIZE, 
		.flags = IORESOURCE_MEM, 
	}, 
};

static struct mn2ws_pcm_desc mn2ws_pcm[] = {
	//ch 0(MAIN)
	{
		.res_regs      = mn2ws_resources, 
		.n_res_regs    = ARRAY_SIZE(mn2ws_resources), 
		
		.hwd = {
			.enabled       = 1, 
			.force_aout    = mn2ws0230_pcm_aio_hwdep_force_aout, 
		}, 
		.play = {
			.enabled       = 1, 
			//0 means allocate buffer from the kernel
			.phys_addr_buf = 0, 
			.size_buf      = MN2WS_PCM_HW_BUF_SIZE, 
			//0 means use default size(= size_buf)
			.size_use_fix  = MN2WS_PCM_HW_USE_SIZE, 
			.init          = mn2ws0230_pcm_aio_play_init, 
			.term          = mn2ws0230_pcm_aio_play_term, 
			.hardware      = mn2ws0230_pcm_aio_play_hardware, 
			.setup         = mn2ws0230_pcm_aio_play_setup_main, 
			.mute          = mn2ws0230_pcm_aio_play_mute, 
			.start         = mn2ws0230_pcm_aio_play_start, 
			.stop          = mn2ws0230_pcm_aio_play_stop, 
			.get_hwptr     = mn2ws0230_pcm_aio_play_get_hwptr, 
			.set_devptr    = mn2ws0230_pcm_aio_play_set_devptr, 
			.wait_hwevent  = mn2ws0230_pcm_aio_play_waitevent, 
			.copy          = mn2ws0230_pcm_aio_play_copy_to_hw, 
			.silence       = mn2ws0230_pcm_aio_play_silence_to_hw, 
		}, 
	}, 
	
	//ch 1(IEC out)
	{
		.res_regs      = mn2ws_resources, 
		.n_res_regs    = ARRAY_SIZE(mn2ws_resources), 
		
		.hwd = {
			.enabled       = 0, 
			.force_aout    = mn2ws0230_pcm_aio_hwdep_force_aout, 
		}, 
		.play = {
			.enabled       = 1, 
			//0 means allocate buffer from the kernel
			.phys_addr_buf = 0, 
			.size_buf      = MN2WS_PCM_HW_BUF_SIZE, 
			//0 means use default size(= size_buf)
			.size_use_fix  = MN2WS_PCM_HW_USE_SIZE, 
			.init          = mn2ws0230_pcm_aio_play_init, 
			.term          = mn2ws0230_pcm_aio_play_term, 
			.hardware      = mn2ws0230_pcm_aio_play_hardware, 
			.setup         = mn2ws0230_pcm_aio_play_setup_iec, 
			.mute          = mn2ws0230_pcm_aio_play_mute, 
			.start         = mn2ws0230_pcm_aio_play_start, 
			.stop          = mn2ws0230_pcm_aio_play_stop, 
			.get_hwptr     = mn2ws0230_pcm_aio_play_get_hwptr, 
			.set_devptr    = mn2ws0230_pcm_aio_play_set_devptr, 
			.wait_hwevent  = mn2ws0230_pcm_aio_play_waitevent, 
			.copy          = mn2ws0230_pcm_aio_play_copy_to_hw, 
			.silence       = mn2ws0230_pcm_aio_play_silence_to_hw, 
		}, 
	}, 
	
	//ch 2(HDMI out)
	{
		.res_regs      = mn2ws_resources, 
		.n_res_regs    = ARRAY_SIZE(mn2ws_resources), 
		
		.hwd = {
			.enabled       = 0, 
			.force_aout    = mn2ws0230_pcm_aio_hwdep_force_aout, 
		}, 
		.play = {
			.enabled       = 1, 
			//0 means allocate buffer from the kernel
			.phys_addr_buf = 0, 
			.size_buf      = MN2WS_PCM_HW_BUF_SIZE_HDMI, 
			//0 means use default size(= size_buf)
			.size_use_fix  = MN2WS_PCM_HW_USE_SIZE_HDMI, 
			.init          = mn2ws0230_pcm_aio_play_init, 
			.term          = mn2ws0230_pcm_aio_play_term, 
			.hardware      = mn2ws0230_pcm_aio_play_hardware_hdmi, 
			.setup         = mn2ws0230_pcm_aio_play_setup_hdmi, 
			.mute          = mn2ws0230_pcm_aio_play_mute, 
			.start         = mn2ws0230_pcm_aio_play_start, 
			.stop          = mn2ws0230_pcm_aio_play_stop, 
			.get_hwptr     = mn2ws0230_pcm_aio_play_get_hwptr, 
			.set_devptr    = mn2ws0230_pcm_aio_play_set_devptr, 
			.wait_hwevent  = mn2ws0230_pcm_aio_play_waitevent, 
			.copy          = mn2ws0230_pcm_aio_play_copy_to_hw, 
			.silence       = mn2ws0230_pcm_aio_play_silence_to_hw, 
		}, 
	}, 
};

static struct mn2ws0230_pcm_aio_desc mn2ws0230_pcm[] = {
	//ch 0(MAIN PCM out)
	{
		.play = {
			.dummy_ch = MN2WS_PCM_AIO_DUMMY_CH_MAIN, 
			.decoutctl_bit = 0, 
			.devsel_bit = 0, 
			.chunk_size = MN2WS_PCM_AIO_CHUNK_SIZE, 
		}, 
	}, 
	
	//ch 1(IEC out)
	{
		.play = {
			.dummy_ch = MN2WS_PCM_AIO_DUMMY_CH_IEC, 
			.decoutctl_bit = 1, 
			.devsel_bit = 2, 
			.chunk_size = MN2WS_PCM_AIO_CHUNK_SIZE, 
		}, 
	}, 
	
	//ch 2(HDMI out)
	{
		.play = {
			.dummy_ch = MN2WS_PCM_AIO_DUMMY_CH_HDMI, 
			.decoutctl_bit = 2, 
			.devsel_bit = 3, 
			.chunk_size = MN2WS_PCM_AIO_CHUNK_SIZE_HDMI, 
		}, 
	}, 
};


const char *get_pcm_dev_name(void)
{
	return MN2WS_PCM_AIO_DEV_NAME;
}

int get_pcm_dev_count(void)
{
	return MN2WS_PCM_AIO_DEV_COUNT;
}

static int mn2ws0230_pcm_aio_hwdep_force_aout(struct mn2ws_pcm_dev *dev)
{
	//do nothing
	
	return 0;
}

static int mn2ws0230_pcm_aio_play_hardware(struct mn2ws_pcm_dev *dev, struct snd_pcm_hardware *h)
{
	DPRINTF("%s\n", __func__);
	
	h->info = (SNDRV_PCM_INFO_MMAP |
		SNDRV_PCM_INFO_MMAP_VALID |
		SNDRV_PCM_INFO_BATCH |
		SNDRV_PCM_INFO_INTERLEAVED |
		//SNDRV_PCM_INFO_NONINTERLEAVED |
		SNDRV_PCM_INFO_BLOCK_TRANSFER); // | 
		//SNDRV_PCM_INFO_PAUSE);
	h->formats = SNDRV_PCM_FMTBIT_S32_BE;
	h->rates = SNDRV_PCM_RATE_48000;
	h->rate_min = 48000;
	h->rate_max = 48000;
	h->channels_min = 2;
	h->channels_max = 2;
	h->buffer_bytes_max = 1024 * 1024;
	h->period_bytes_min = 256;
	h->period_bytes_max = 256 * 1024;
	h->periods_min = 2;
	h->periods_max = 1024;
	h->fifo_size = 0;
	
	return 0;
}

static int mn2ws0230_pcm_aio_play_hardware_hdmi(struct mn2ws_pcm_dev *dev, struct snd_pcm_hardware *h)
{
	DPRINTF("%s\n", __func__);
	
	h->info = (SNDRV_PCM_INFO_MMAP |
		SNDRV_PCM_INFO_MMAP_VALID |
		SNDRV_PCM_INFO_BATCH |
		SNDRV_PCM_INFO_INTERLEAVED |
		//SNDRV_PCM_INFO_NONINTERLEAVED |
		SNDRV_PCM_INFO_BLOCK_TRANSFER); // | 
		//SNDRV_PCM_INFO_PAUSE);
	h->formats = SNDRV_PCM_FMTBIT_S32_BE;
	h->rates = SNDRV_PCM_RATE_48000;
	h->rate_min = 48000;
	h->rate_max = 48000;
	h->channels_min = 8;
	h->channels_max = 8;
	h->buffer_bytes_max = 1024 * 1024;
	h->period_bytes_min = 256;
	h->period_bytes_max = 256 * 1024;
	h->periods_min = 2;
	h->periods_max = 1024;
	h->fifo_size = 0;
	
	return 0;
}

static int mn2ws0230_pcm_aio_play_init(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	u32 v;
	
	//Stop all output
	v = pcm_readl(mmp, MN2WS_PCMOCTL_0);
	v &= ~0x1;
	pcm_writel(mmp, v, MN2WS_PCMOCTL_0);
	
	v = pcm_readl(mmp, MN2WS_IECOCTL_0);
	v &= ~0x1;
	pcm_writel(mmp, v, MN2WS_IECOCTL_0);
	
	v = pcm_readl(mmp, MN2WS_HDMIOCTL_0);
	v &= ~0x1;
	pcm_writel(mmp, v, MN2WS_HDMIOCTL_0);
	
	v = pcm_readl(mmp, MN2WS_SRCCTL_0);
	v &= ~0x1;
	pcm_writel(mmp, v, MN2WS_SRCCTL_0);
	
	v = pcm_readl(mmp, MN2WS_SRCCTL_1);
	v &= ~0x1;
	pcm_writel(mmp, v, MN2WS_SRCCTL_1);
	
	//Ready to change the mode
	pcm_writel(mmp, 0x0, MN2WS_ADECOUTCTL);
	pcm_writel(mmp, 0x0, MN2WS_SYSCTRCMDA);
	
	return 0;
}

static int mn2ws0230_pcm_aio_play_term(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	u32 v;
	
	//Set mode
	// 0   : Mode : Tunnel mode
	pcm_writel(mmp, 0x0, MN2WS_APCMOUT_MODE(dev->ch));
	
	//Select nothing
	pcm_writel(mmp, 0x0, MN2WS_ADECOUTCTL);
	pcm_writel(mmp, 0x0, MN2WS_SYSCTRCMDA);
	
	//Send
	pcm_writel(mmp, 0x1, MN2WS_APCMOUT_SEMAPH(dev->ch));
	
	v = pcm_readl(mmp, MN2WS_APCMOUT_SEMAPH(dev->ch));
	while (v != 0) {
		v = pcm_readl(mmp, MN2WS_APCMOUT_SEMAPH(dev->ch));
	}
	
	return 0;
}

/**
 * ꡼׶ػ(main out)
 */
static int mn2ws0230_pcm_aio_play_setup_main(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	
	DPRINTF("setup the playback %d(main).\n", dev->ch);
	
	//Set non-tunnel mode
	mn2ws0230_pcm_aio_play_setup_mode(dev);
	
	
	//Set output signal
	// 0   : OUTPTENB  : enable
	// 1- 3: OUTMODE   : clock ch0
	// 4- 5: DACCKSEL  : aclk * 1
	// 6- 7: LR_SWITCH : Stereo
	// 8- 9: MAINDS    : off
	//10   : EXTIFSEL  : 36.864MHz
	//11   : EXCKSEL   : Internal PLL
	//12-13: QUANT_BIT : 24bits
	//                   0: 24bits
	//                   1: 20bits
	//                   2: 16bits
	//                   3: 32bits
	//14-15: IF_FORMAT : I2S
	//                   0: Right justified
	//                   1: Left justified
	//                   3: I2S
	//16-19: ACTCKSEL  : APLLA1
	//26-30: AFADETIME : 0
	//31   : TERMUTE   : off
	pcm_writel(mmp, 0x0000c021, MN2WS_APCMOUT_SEND_XOCTL(dev->ch));
	
	//Set output freq
	// 0- 8: freq : 48kHz
	//              0: 48kHz
	//              1: 96kHz
	//              2: 192kHz
	//              3: 32kHz
	//              4: 44kHz
	//              5: 88kHz
	pcm_writel(mmp, 0x0, MN2WS_APCMOUT_SEND_XOFS(dev->ch));
	
	//Setup buffer of the DMA
	mn2ws0230_pcm_aio_play_setup_buffer(dev);
	
	//Setup the DMA
	mn2ws0230_pcm_aio_play_setup_dma(dev);
	
	mn2ws0230_pcm_aio_play_mute(dev);
	
	return 0;
}

/**
 * ꡼׶ػ(IEC out)
 */
static int mn2ws0230_pcm_aio_play_setup_iec(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	
	DPRINTF("setup the playback %d(IEC).\n", dev->ch);
	
	//Set non-tunnel mode
	mn2ws0230_pcm_aio_play_setup_mode(dev);
	
	
	//Set output signal
	// 0   : OUTPTENB  : enable
	// 1- 3: OUTMODE   : clock ch0
	// 6- 7: LR_SWITCH : Stereo
	// 8- 9: IECDS     : off
	//10   : EXTIFSEL  : 36.864MHz
	//11   : EXCKSEL   : Internal PLL
	//12-13: QUANT_BIT : 24bits
	//                   0: 24bits
	//                   1: 20bits
	//                   2: 16bits
	//                   3: 32bits
	//14-15: IF_FORMAT : I2S
	//                   0: Right justified
	//                   1: Left justified
	//                   3: I2S
	//16-19: ACTCKSEL  : APLLA1
	//26-30: AFADETIME : 0
	//31   : TERMUTE   : off
	pcm_writel(mmp, 0x0000c001, MN2WS_APCMOUT_SEND_XOCTL(dev->ch));
	
	//Set output freq
	// 0- 8: freq : 0: 48kHz
	//              1: 96kHz
	//              2: 192kHz
	//              3: 32kHz
	//              4: 44kHz
	//              5: 88kHz
	pcm_writel(mmp, 0x0, MN2WS_APCMOUT_SEND_XOFS(dev->ch));
	
	//Setup buffer of the DMA
	mn2ws0230_pcm_aio_play_setup_buffer(dev);
	
	//Setup the DMA
	mn2ws0230_pcm_aio_play_setup_dma(dev);
	
	mn2ws0230_pcm_aio_play_mute(dev);
	
	return 0;
}

/**
 * ꡼׶ػ(HDMI out)
 */
static int mn2ws0230_pcm_aio_play_setup_hdmi(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	u32 v;
	
	DPRINTF("setup the playback %d(HDMI).\n", dev->ch);
	
	//Set non-tunnel mode
	mn2ws0230_pcm_aio_play_setup_mode(dev);
	
	
	//Set output signal
	// 0   : OUTPTENB  : enable
	// 1- 3: OUTMODE   : clock ch
	// 8- 9: QUANT_BIT : 24bits
	//                   0: 24bits
	//                   1: 20bits
	//                   2: 16bits
	//                   3: 32bits
	//11   : EXCKSEL   : Internal PLL
	//14   : EXTIFSEL  : 36.864MHz
	//16-19: ACTCKSEL  : APLLA1
	//21   : HDMIMUTE  : off
	//26-30: AFADETIME : 0
	//31   : TERMUTE   : off
	v = 0x00000001;
	v |= (dev->ch << 1);
	pcm_writel(mmp, v, MN2WS_APCMOUT_SEND_XOCTL(dev->ch));
	
	//Set output freq
	// 0- 8: freq : 48kHz
	//              0: 48kHz
	//              1: 96kHz
	//              2: 192kHz
	//              3: 32kHz
	//              4: 44kHz
	//              5: 88kHz
	pcm_writel(mmp, 0x0, MN2WS_APCMOUT_SEND_XOFS(dev->ch));
	
	//Set output PCM for HDMI
	// 3- 4: DACCKSEL    : aclk * 1
	//16-17: LR_SWITCH   : Stereo
	//18   : CLFE_SWITCH : Normal
	//20-21: IF_FORMAT   : I2S
	//                     0: Right justified
	//                     1: Left justified
	//                     3: I2S
	pcm_writel(mmp, 0x00300010, MN2WS_APCMOUT_SEND_HDMIPCMOCTL);
	
	//Set output IEC for HDMI
	//16-17: LR_SWITCH   : Stereo
	//20   : HDMIO       : Enable
	//22-23: HDMIPS      : Pause burst(Swithing), Pause burst(Pausing)
	//24   : LOCKSEL     : Non-stop
	//pcm_writel(mmp, 0x00100000, MN2WS_APCMOUT_SEND_HDMIIECOCTL);
	
	//PCM or IEC
	// 0   : .... : PCM out
	pcm_writel(mmp, 0x00000000, MN2WS_APCMOUT_SEND_HDMIFMTSEL);
	
	
	//Setup buffer of the DMA
	mn2ws0230_pcm_aio_play_setup_buffer(dev);
	
	//Setup the DMA
	mn2ws0230_pcm_aio_play_setup_dma(dev);
	
	mn2ws0230_pcm_aio_play_mute(dev);
	
	return 0;
}

/**
 * ꡼׶ػ
 */
static int mn2ws0230_pcm_aio_play_setup_mode(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	struct mn2ws0230_pcm_aio_desc *aio_desc = dev->desc->private_data;
	u32 st, ed, dm_st, dm_ed, v;
	
	DPRINTF("setup mode of playback %d.\n", dev->ch);

	st = dev->play.paddr_buf;
	ed = st + dev->play.size_buf_use;
	dm_st = ed;
	dm_ed = dm_st + 0x4000;
	
	//Set chunk size: 0x100(fixed by HW)
	pcm_writel(mmp, 0x100, MN2WS_APCMOUT_SAMP(dev->ch));
	
	//Dummy channel
	pcm_writel(mmp, dm_st, 
		MN2WS_APCMOUT_SEND_STADDR(aio_desc->play.dummy_ch));
	pcm_writel(mmp, dm_ed, 
		MN2WS_APCMOUT_SEND_ENDADDR(aio_desc->play.dummy_ch));
	
	//Dummy FS: 0x0(fixed by HW)
	pcm_writel(mmp, 0x0, MN2WS_APCMOUT_SEND_DUMMY0FS);
	
	
	//Set mode
	// 0   : Mode : Non-tunnel mode
	pcm_writel(mmp, 0x1, MN2WS_APCMOUT_MODE(dev->ch));
	
	//Select dec ch
	v = pcm_readl(mmp, MN2WS_ADECOUTCTL);
	v |= (1 << aio_desc->play.decoutctl_bit);
	pcm_writel(mmp, v, MN2WS_ADECOUTCTL);
	
	v = pcm_readl(mmp, MN2WS_SYSCTRCMDA);
	v |= (1 << aio_desc->play.decoutctl_bit);
	pcm_writel(mmp, v, MN2WS_SYSCTRCMDA);
	
	v = pcm_readl(mmp, MN2WS_APCMOUT_SEMAPH(dev->ch));
	while (v != 0) {
		v = pcm_readl(mmp, MN2WS_APCMOUT_SEMAPH(dev->ch));
	}
	
	return 0;
}

/**
 * ꡼׶ػ
 */
static int mn2ws0230_pcm_aio_play_setup_buffer(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	struct mn2ws0230_pcm_aio_desc *aio_desc = dev->desc->private_data;
	u32 st, ed, v;
	
	DPRINTF("setup buffer of playback %d.\n", dev->ch);

	st = dev->play.paddr_buf;
	ed = st + dev->play.size_buf_use;
	
	//DMA buffer address
	pcm_writel(mmp, st, MN2WS_APCMOUT_SEND_STADDR(dev->ch));
	pcm_writel(mmp, ed, MN2WS_APCMOUT_SEND_ENDADDR(dev->ch));
	
	//Select port for configuration
	v = (1 << aio_desc->play.devsel_bit);
	pcm_writel(mmp, v, MN2WS_APCMOUT_DEVSEL_CONF(dev->ch));
	
	return 0;
}

/**
 * ꡼׶ػ
 */
static int mn2ws0230_pcm_aio_play_setup_dma(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	struct mn2ws0230_pcm_aio_desc *aio_desc = dev->desc->private_data;
	u32 st, ed, rd, v;
	int i;
	
	DPRINTF("setup DMA of playback %d.\n", dev->ch);
	
	st = dev->play.paddr_buf;
	ed = st + dev->play.size_buf_use;

	//Send zero data(max 10-chunks)
	for (i = 0; i < 10; i++) {
		rd = pcm_readl(mmp, MN2WS_APCMOUT_SEND_RDSRC(dev->ch));
		if (rd == 0) {
			rd = st;
		}
		pcm_writel(mmp, rd, MN2WS_APCMOUT_SEND_WRSRC(dev->ch));
		
		//Select port for output
		v = (1 << aio_desc->play.devsel_bit);
		pcm_writel(mmp, v, MN2WS_APCMOUT_DEVSEL_SEND(dev->ch));
		
		//Send chunk
		pcm_writel(mmp, 0x1, MN2WS_APCMOUT_SEMAPH(dev->ch));
		
		v = pcm_readl(mmp, MN2WS_APCMOUT_SEMAPH(dev->ch));
		while (v != 0) {
			v = pcm_readl(mmp, MN2WS_APCMOUT_SEMAPH(dev->ch));
		}
		
		if (mn2ws0230_pcm_aio_play_get_hwptr(dev) != 0) {
			break;
		}
	}
	
	return 0;
}

/**
 * ꡼׶ػ
 */
static int mn2ws0230_pcm_aio_play_mute(struct mn2ws_pcm_dev *dev)
{
	//Do nothing
	
	return 0;
}

/**
 * ꡼׶ػ
 */
static int mn2ws0230_pcm_aio_play_start(struct mn2ws_pcm_dev *dev)
{
	//Do nothing
	
	return 0;
}

/**
 * ꡼׶ػ
 */
static int mn2ws0230_pcm_aio_play_stop(struct mn2ws_pcm_dev *dev)
{
	//Do nothing
	
	return 0;
}

/**
 * PCM  HW ɤ߼äƤХåեˤĤơ
 * ߤ HW ɤ߼֤롣
 * 
 * ХåեƬ 0 Ȥ롣
 * 
 * @param dev PCM ǥХ
 * @return HW θߤɤ߼
 */
static unsigned long mn2ws0230_pcm_aio_play_get_hwptr(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	
	return pcm_readl(mmp, MN2WS_APCMOUT_SEND_RDSRC(dev->ch)) - 
		pcm_readl(mmp, MN2WS_APCMOUT_SEND_STADDR(dev->ch));
}

/**
 * PCM  HW ɤ߼äƤХåեˤĤơ
 * ߤΥǥХν񤭹֤߰ꤹ롣
 * 
 * ХåեƬ 0 Ȥ롣
 * 
 * @param dev PCM ǥХ
 * @param ptr ǥХθߤν񤭹߰
 */
static void mn2ws0230_pcm_aio_play_set_devptr(struct mn2ws_pcm_dev *dev, unsigned long ptr)
{
	//Do nothing
}

static int mn2ws0230_pcm_aio_play_waitevent(struct mn2ws_pcm_dev *dev)
{
	ktime_t expires;
	
	expires = ktime_add_ns(ktime_get(), 1000000);
	set_current_state(TASK_INTERRUPTIBLE);
	schedule_hrtimeout_range(&expires, 3000000, HRTIMER_MODE_ABS);
	
	return 0;
}

static ssize_t mn2ws0230_pcm_aio_play_copy_to_hw(struct mn2ws_pcm_dev *dev, struct ringbuf *r, size_t s)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	struct mn2ws0230_pcm_aio_desc *aio_desc = dev->desc->private_data;
	u32 rd, rd_off, v;
	ssize_t n;
	
	v = pcm_readl(mmp, MN2WS_APCMOUT_SEMAPH(dev->ch));
	if (v != 0) {
		//busy
		return 0;
	}
	
	//NOTICE: APCMOUT must start writing from read pointer...
	rd_off = mn2ws0230_pcm_aio_play_get_hwptr(dev);
	set_rp_ringbuf(&dev->play.hw, rd_off);
	set_wp_ringbuf(&dev->play.hw, rd_off);
	
	if (s < aio_desc->play.chunk_size) {
		//too short
		return 0;
	}
	
	//NOTICE: must write once, and size must be same as chunk_size...
	n = copy_ringbuf(&dev->play.hw, r, aio_desc->play.chunk_size);
	if (n != aio_desc->play.chunk_size) {
		PRINTF_WARN("ch%d_p: cannot copy to chunk_size(0x%x).\n", 
			dev->ch, aio_desc->play.chunk_size);
	}
	
	//Send 1-chunk
	rd = pcm_readl(mmp, MN2WS_APCMOUT_SEND_RDSRC(dev->ch));
	pcm_writel(mmp, rd, MN2WS_APCMOUT_SEND_WRSRC(dev->ch));
	
	//Set port for output
	v = (1 << aio_desc->play.devsel_bit);
	pcm_writel(mmp, v, MN2WS_APCMOUT_DEVSEL_SEND(dev->ch));
	
	//Send chunk
	pcm_writel(mmp, 0x1, MN2WS_APCMOUT_SEMAPH(dev->ch));
	
	return aio_desc->play.chunk_size;
}

static ssize_t mn2ws0230_pcm_aio_play_silence_to_hw(struct mn2ws_pcm_dev *dev, size_t s)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	struct mn2ws0230_pcm_aio_desc *aio_desc = dev->desc->private_data;
	u32 rd, rd_off, v;
	ssize_t n;
	
	v = pcm_readl(mmp, MN2WS_APCMOUT_SEMAPH(dev->ch));
	if (v != 0) {
		//busy
		return 0;
	}
	
	//NOTICE: APCMOUT must start writing from read pointer...
	rd_off = mn2ws0230_pcm_aio_play_get_hwptr(dev);
	set_rp_ringbuf(&dev->play.hw, rd_off);
	set_wp_ringbuf(&dev->play.hw, rd_off);
	
	if (s < aio_desc->play.chunk_size) {
		//too short
		return 0;
	}

	//NOTICE: must write once, and size must be same as chunk_size...
	n = write_silent_ringbuf(&dev->play.hw, aio_desc->play.chunk_size);
	if (n != aio_desc->play.chunk_size) {
		PRINTF_WARN("ch%d_p: cannot copy to chunk_size(0x%x).\n", 
			dev->ch, aio_desc->play.chunk_size);
	}
	
	//Send 1-chunk
	rd = pcm_readl(mmp, MN2WS_APCMOUT_SEND_RDSRC(dev->ch));
	pcm_writel(mmp, rd, MN2WS_APCMOUT_SEND_WRSRC(dev->ch));
	
	//Set port for output
	v = (1 << aio_desc->play.devsel_bit);
	pcm_writel(mmp, v, MN2WS_APCMOUT_DEVSEL_SEND(dev->ch));
	
	//Send chunk
	pcm_writel(mmp, 0x1, MN2WS_APCMOUT_SEMAPH(dev->ch));
	
	return aio_desc->play.chunk_size;
}


static int __init mn2ws0230_pcm_aio_init_module(void)
{
	int i;
	
	if (ARRAY_SIZE(mn2ws_pcm) != ARRAY_SIZE(mn2ws0230_pcm)) {
		PRINTF_WARN("BUG!! SoC descriptor size is illegal."
			"(pcm:%d, soc_pcm:%d)\n", 
			ARRAY_SIZE(mn2ws_pcm), ARRAY_SIZE(mn2ws0230_pcm));
		snd_BUG();
	}
	
	for (i = 0; i < ARRAY_SIZE(mn2ws_pcm); i++) {
		mn2ws_pcm[i].private_data = &mn2ws0230_pcm[i];
	}
	
	return mn2ws_pcm_init_module(mn2ws_pcm);
}

static void __exit mn2ws0230_pcm_aio_exit_module(void)
{
	mn2ws_pcm_exit_module(mn2ws_pcm);
}

module_init(mn2ws0230_pcm_aio_init_module);
module_exit(mn2ws0230_pcm_aio_exit_module);
